from __future__ import annotations

import os
import shutil
import subprocess
import tempfile
from pathlib import Path
from typing import Dict, List


CleanupReport = Dict[str, List[str]]


class CleanupError(RuntimeError):
    """Raised when a cleanup action fails."""


def _delete_path(path: Path, report: CleanupReport, category: str) -> None:
    try:
        if path.is_dir():
            shutil.rmtree(path)
        else:
            path.unlink(missing_ok=True)
        report[category].append(str(path))
    except Exception as exc:
        report.setdefault("errors", []).append(f"{path}: {exc}")


def clean_temp_directories() -> CleanupReport:
    report: CleanupReport = {"deleted": []}

    temp_root = Path(tempfile.gettempdir())
    windows_temp = Path("C:/Windows/Temp")
    user_temp = temp_root

    for directory in {temp_root, windows_temp, user_temp}:
        if directory.exists():
            for child in directory.iterdir():
                _delete_path(child, report, "deleted")

    return report


def purge_prefetch() -> CleanupReport:
    report: CleanupReport = {"deleted": []}
    prefetch_dir = Path("C:/Windows/Prefetch")
    if not prefetch_dir.exists():
        return report

    for file in prefetch_dir.iterdir():
        if file.is_file():
            _delete_path(file, report, "deleted")
    return report


def clear_windows_update_cache() -> CleanupReport:
    report: CleanupReport = {"deleted": []}
    distribution_dir = Path("C:/Windows/SoftwareDistribution/Download")
    if not distribution_dir.exists():
        return report

    for item in distribution_dir.iterdir():
        _delete_path(item, report, "deleted")
    return report


def run_storage_sense() -> CleanupReport:
    """
    Trigger Windows storage sense to clean up old files.

    Returns the textual output emitted by the command.
    """
    report: CleanupReport = {"output": []}
    command = [
        "powershell",
        "-NoProfile",
        "-ExecutionPolicy",
        "Bypass",
        "-Command",
        "Start-Process -FilePath cleanmgr.exe -ArgumentList '/sagerun:1' -Wait",
    ]
    try:
        completed = subprocess.run(
            command,
            capture_output=True,
            text=True,
            check=False,
        )
        if completed.stdout:
            report["output"].append(completed.stdout.strip())
        if completed.stderr:
            report.setdefault("errors", []).append(completed.stderr.strip())
    except FileNotFoundError:
        report.setdefault("errors", []).append("cleanmgr.exe not available on this system.")
    except Exception as exc:
        report.setdefault("errors", []).append(str(exc))
    return report


def cleanup_browser_caches() -> CleanupReport:
    """
    Remove cache folders for common browsers (Edge, Chrome, Firefox).
    Only deletes files inside the cache directories, not the directories themselves.
    """
    report: CleanupReport = {"deleted": []}

    local_appdata = Path(os.environ.get("LOCALAPPDATA", ""))
    appdata = Path(os.environ.get("APPDATA", ""))

    cache_targets = [
        local_appdata / "Microsoft" / "Edge" / "User Data" / "Default" / "Cache",
        local_appdata / "Google" / "Chrome" / "User Data" / "Default" / "Cache",
        appdata / "Mozilla" / "Firefox" / "Profiles",
    ]

    for target in cache_targets:
        if not target.exists():
            continue
        if target.name == "Profiles":  # Firefox structure
            for profile_dir in target.iterdir():
                cache_dir = profile_dir / "cache2"
                if cache_dir.exists():
                    for item in cache_dir.glob("**/*"):
                        if item.is_file():
                            _delete_path(item, report, "deleted")
        else:
            for item in target.glob("**/*"):
                if item.is_file():
                    _delete_path(item, report, "deleted")
    return report

